<?php

namespace App\Livewire\Host;

use Livewire\Component;
use App\Models\PickupLocation;

class PickupLocationsManager extends Component
{
    public $locations;
    public $showForm = false;
    public $editMode = false;
    public $locationId;

    public $name;
    public $address;
    public $type = 'general';
    public $latitude;
    public $longitude;
    public $is_active = true;
    public $typeFilter = '';

    public function mount()
    {
        // Check if user has permission to create custom locations
        if (!auth()->user()->can_create_custom_locations) {
            session()->flash('error', 'You do not have permission to create custom pickup locations. Please contact the administrator.');
            return redirect()->route('host.listings');
        }

        $this->loadLocations();
    }

    public function loadLocations()
    {
        // Only show locations created by the current host
        $query = PickupLocation::where('created_by', auth()->id())
            ->orderBy('created_at', 'desc');

        if ($this->typeFilter) {
            $query->where('type', $this->typeFilter);
        }

        $this->locations = $query->get();
    }

    public function updatedTypeFilter()
    {
        $this->loadLocations();
    }

    public function create()
    {
        $this->resetForm();
        $this->showForm = true;
        $this->editMode = false;
    }

    public function edit($id)
    {
        // Ensure the location belongs to the current host
        $location = PickupLocation::where('created_by', auth()->id())->findOrFail($id);
        $this->locationId = $location->id;
        $this->name = $location->name;
        $this->address = $location->address;
        $this->type = $location->type;
        $this->latitude = $location->latitude;
        $this->longitude = $location->longitude;
        $this->is_active = $location->is_active;
        $this->showForm = true;
        $this->editMode = true;
    }

    public function save()
    {
        // Check location limit (3 locations per host) - only for new locations
        if (!$this->editMode) {
            $currentCount = PickupLocation::where('created_by', auth()->id())->count();
            if ($currentCount >= 3) {
                $this->addError('name', 'You can only create up to 3 custom pickup locations. Please delete an existing location to add a new one.');
                return;
            }
        }

        $this->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string|max:500',
            'type' => 'required|in:airport,hotel,public_spot,office,general',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);

        $data = [
            'name' => $this->name,
            'address' => $this->address,
            'type' => $this->type,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'is_active' => $this->is_active,
        ];

        if ($this->editMode) {
            // Ensure the location belongs to the current host
            $location = PickupLocation::where('created_by', auth()->id())->findOrFail($this->locationId);
            $location->update($data);
            session()->flash('message', 'Pickup location updated successfully!');
        } else {
            $data['created_by'] = auth()->id();
            PickupLocation::create($data);
            session()->flash('message', 'Pickup location created successfully!');
        }

        $this->loadLocations();
        $this->showForm = false;
        $this->resetForm();
    }

    public function toggleStatus($id)
    {
        // Ensure the location belongs to the current host
        $location = PickupLocation::where('created_by', auth()->id())->findOrFail($id);
        $location->update(['is_active' => !$location->is_active]);
        $this->loadLocations();
        session()->flash('message', 'Location status updated!');
    }

    public function delete($id)
    {
        // Ensure the location belongs to the current host
        PickupLocation::where('created_by', auth()->id())->findOrFail($id)->delete();
        $this->loadLocations();
        session()->flash('message', 'Pickup location deleted successfully!');
    }

    public function resetForm()
    {
        $this->name = '';
        $this->address = '';
        $this->type = 'general';
        $this->latitude = null;
        $this->longitude = null;
        $this->is_active = true;
    }

    public function render()
    {
        return view('livewire.host.pickup-locations-manager')->layout('layouts.app');
    }
}
