<?php

namespace App\Livewire\Host;

use Livewire\Component;
use App\Models\Booking;
use Carbon\Carbon;

class BookingsManager extends Component
{
    public $bookings;
    public $filter = 'all'; // all, pending, confirmed, completed, cancelled
    public $dateFilter = 'all'; // all, today, week, month, custom
    public $startDate = null;
    public $endDate = null;
    public $search = ''; // Search query

    // Analytics
    public $totalBookings = 0;
    public $confirmedBookings = 0;
    public $completedBookings = 0;
    public $totalRevenue = 0;

    public function mount()
    {
        $this->loadBookings();
        $this->loadAnalytics();
    }

    public function loadBookings()
    {
        // Get only bookings for the host's listings
        $query = Booking::with(['listing.location', 'listing.rentalType', 'user', 'payment', 'addOns', 'pickupLocation', 'dropoffLocation'])
            ->whereHas('listing', function($q) {
                $q->where('user_id', auth()->id());
            })
            ->orderBy('created_at', 'desc');

        // Apply search filter
        if (!empty($this->search)) {
            $query->where(function($q) {
                $q->where('id', 'like', '%' . $this->search . '%')
                  ->orWhere('invoice_number', 'like', '%' . $this->search . '%')
                  ->orWhereHas('user', function($q) {
                      $q->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('email', 'like', '%' . $this->search . '%');
                  })
                  ->orWhereHas('listing', function($q) {
                      $q->where('title', 'like', '%' . $this->search . '%');
                  });
            });
        }

        // Apply status filter
        if ($this->filter !== 'all') {
            $query->where('status', $this->filter);
        }

        // Apply date filter
        if ($this->dateFilter !== 'all') {
            switch ($this->dateFilter) {
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $query->whereBetween('created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()]);
                    break;
                case 'custom':
                    if ($this->startDate && $this->endDate) {
                        $query->whereBetween('created_at', [$this->startDate, $this->endDate]);
                    }
                    break;
            }
        }

        $this->bookings = $query->get();
    }

    public function loadAnalytics()
    {
        $query = Booking::whereHas('listing', function($q) {
            $q->where('user_id', auth()->id());
        });

        // Apply date filter to analytics too
        if ($this->dateFilter !== 'all') {
            switch ($this->dateFilter) {
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $query->whereBetween('created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()]);
                    break;
                case 'custom':
                    if ($this->startDate && $this->endDate) {
                        $query->whereBetween('created_at', [$this->startDate, $this->endDate]);
                    }
                    break;
            }
        }

        $this->totalBookings = (clone $query)->count();
        $this->confirmedBookings = (clone $query)->where('status', 'confirmed')->count();
        $this->completedBookings = (clone $query)->where('status', 'completed')->count();
        $this->totalRevenue = (clone $query)->whereIn('status', ['confirmed', 'completed'])->sum('total_price');
    }

    public function updateFilter()
    {
        $this->loadBookings();
    }

    public function updateDateFilter()
    {
        $this->loadBookings();
        $this->loadAnalytics();
    }

    public function updatedSearch()
    {
        $this->loadBookings();
    }

    public function applyCustomDate()
    {
        if ($this->startDate && $this->endDate) {
            $this->dateFilter = 'custom';
            $this->loadBookings();
            $this->loadAnalytics();
        }
    }

    public function exportCsv()
    {
        // Get only bookings for the host's listings
        $query = Booking::with(['listing.location', 'listing.rentalType', 'user', 'addOns', 'pickupLocation', 'dropoffLocation'])
            ->whereHas('listing', function($q) {
                $q->where('user_id', auth()->id());
            })
            ->orderBy('created_at', 'desc');

        // Apply status filter
        if ($this->filter !== 'all') {
            $query->where('status', $this->filter);
        }

        // Apply date filter
        if ($this->dateFilter !== 'all') {
            switch ($this->dateFilter) {
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $query->whereBetween('created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()]);
                    break;
                case 'custom':
                    if ($this->startDate && $this->endDate) {
                        $query->whereBetween('created_at', [$this->startDate, $this->endDate]);
                    }
                    break;
            }
        }

        $bookings = $query->get();

        $filename = 'my_bookings_export_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($bookings) {
            $file = fopen('php://output', 'w');

            // Add CSV headers
            fputcsv($file, [
                'Invoice Number',
                'Guest Name',
                'Guest Email',
                'Listing Title',
                'Check In',
                'Check Out',
                'Guests',
                'Total Price (RM)',
                'Status',
                'Booking Date'
            ]);

            // Add data rows
            foreach ($bookings as $booking) {
                fputcsv($file, [
                    $booking->invoice_number,
                    $booking->user->name ?? 'N/A',
                    $booking->user->email ?? 'N/A',
                    $booking->listing->title ?? 'N/A',
                    $booking->check_in ? Carbon::parse($booking->check_in)->format('Y-m-d') : 'N/A',
                    $booking->check_out ? Carbon::parse($booking->check_out)->format('Y-m-d') : 'N/A',
                    $booking->guests,
                    number_format($booking->total_price, 2),
                    ucfirst($booking->status),
                    $booking->created_at->format('Y-m-d H:i:s')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function render()
    {
        return view('livewire.host.bookings-manager')->layout('layouts.app');
    }
}
