<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\Setting;
use App\Models\OutgoingWebhookConfig;
use App\Models\EmailTemplate;

class Settings extends Component
{
    // Payment Gateway Settings
    public $bayarcash_portal = '';
    public $bayarcash_pat = '';
    public $bayarcash_secret = '';
    public $bayarcash_mode = 'sandbox';

    // Payment Gateway Fee Settings
    public $payment_gateway_fee_type = 'percentage';
    public $payment_gateway_fee_fixed = 0;
    public $payment_gateway_fee_percentage = 0;

    // Platform Fee Settings
    public $platform_fee_type = 'percentage';
    public $platform_fee_fixed = 0;
    public $platform_fee_percentage = 0;
    public $platform_fee_charged_to = 'guest';
    public $featured_listing_fee = 0;

    // Outgoing Webhook Settings
    public $webhook_enabled = false;
    public $webhook_url = '';
    public $webhook_payment_success = false;
    public $webhook_payment_failed = false;
    public $webhook_booking_created = false;
    public $webhook_booking_cancelled = false;
    public $webhook_secret_key = '';

    // Email/SMTP Settings
    public $mail_mailer = 'smtp';
    public $mail_host = '';
    public $mail_port = 587;
    public $mail_username = '';
    public $mail_password = '';
    public $mail_encryption = 'tls';
    public $mail_from_address = '';
    public $mail_from_name = '';

    // AWS SES Settings
    public $aws_access_key_id = '';
    public $aws_secret_access_key = '';
    public $aws_default_region = 'us-east-1';

    // Receipt Settings
    public $receipt_template = 'professional';
    public $receipt_company_name = '';
    public $receipt_company_address = '';
    public $receipt_company_phone = '';
    public $receipt_company_email = '';
    public $receipt_website = '';
    public $receipt_footer_text = '';
    public $receipt_show_logo = true;

    // Email Templates
    public $templates;
    public $editingTemplate = null;
    public $template_name;
    public $template_subject;
    public $template_body;
    public $template_enabled;
    public $showEditModal = false;
    public $showPreviewModal = false;

    public function mount()
    {
        // Load current settings
        $this->bayarcash_portal = Setting::get('bayarcash_portal', '');
        $this->bayarcash_pat = Setting::get('bayarcash_pat', '');
        $this->bayarcash_secret = Setting::get('bayarcash_secret', '');
        $this->bayarcash_mode = Setting::get('bayarcash_mode', 'sandbox');
        $this->payment_gateway_fee_type = Setting::get('payment_gateway_fee_type', 'percentage');
        $this->payment_gateway_fee_fixed = Setting::get('payment_gateway_fee_fixed', 0);
        $this->payment_gateway_fee_percentage = Setting::get('payment_gateway_fee_percentage', 0);
        $this->platform_fee_type = Setting::get('platform_fee_type', 'percentage');
        $this->platform_fee_fixed = Setting::get('platform_fee_fixed', 0);
        $this->platform_fee_percentage = Setting::get('platform_fee_percentage', 0);
        $this->platform_fee_charged_to = Setting::get('platform_fee_charged_to', 'guest');
        $this->featured_listing_fee = Setting::get('featured_listing_fee', 0);

        // Load webhook settings
        $webhookConfig = OutgoingWebhookConfig::getConfig();
        $this->webhook_enabled = $webhookConfig->enabled;
        $this->webhook_url = $webhookConfig->webhook_url ?? '';
        $this->webhook_payment_success = $webhookConfig->payment_success;
        $this->webhook_payment_failed = $webhookConfig->payment_failed;
        $this->webhook_booking_created = $webhookConfig->booking_created;
        $this->webhook_booking_cancelled = $webhookConfig->booking_cancelled;
        $this->webhook_secret_key = $webhookConfig->secret_key ?? '';

        // Load email settings
        $this->mail_mailer = Setting::get('mail_mailer', 'smtp');
        $this->mail_host = Setting::get('mail_host', '');
        $this->mail_port = Setting::get('mail_port', 587);
        $this->mail_username = Setting::get('mail_username', '');
        $this->mail_password = Setting::get('mail_password', '');
        $this->mail_encryption = Setting::get('mail_encryption', 'tls');
        $this->mail_from_address = Setting::get('mail_from_address', '');
        $this->mail_from_name = Setting::get('mail_from_name', '');

        // Load AWS SES settings
        $this->aws_access_key_id = Setting::get('aws_access_key_id', '');
        $this->aws_secret_access_key = Setting::get('aws_secret_access_key', '');
        $this->aws_default_region = Setting::get('aws_default_region', 'us-east-1');

        // Load receipt settings
        $this->receipt_template = Setting::get('receipt_template', 'professional');
        $this->receipt_company_name = Setting::get('receipt_company_name', '');
        $this->receipt_company_address = Setting::get('receipt_company_address', '');
        $this->receipt_company_phone = Setting::get('receipt_company_phone', '');
        $this->receipt_company_email = Setting::get('receipt_company_email', '');
        $this->receipt_website = Setting::get('receipt_website', '');
        $this->receipt_footer_text = Setting::get('receipt_footer_text', 'Thank you for your business!');
        $this->receipt_show_logo = Setting::get('receipt_show_logo', true);

        // Load email templates
        $this->loadTemplates();
    }

    public function loadTemplates()
    {
        $this->templates = EmailTemplate::orderBy('name')->get();
    }

    public function editTemplate($id)
    {
        $template = EmailTemplate::findOrFail($id);
        $this->editingTemplate = $template->id;
        $this->template_name = $template->name;
        $this->template_subject = $template->subject;
        $this->template_body = $template->body;
        $this->template_enabled = $template->enabled;
        $this->showEditModal = true;
    }

    public function previewTemplate($id)
    {
        $template = EmailTemplate::findOrFail($id);
        $this->template_name = $template->name;
        $this->template_subject = $template->subject;
        $this->template_body = $template->body;
        $this->showPreviewModal = true;
    }

    public function saveTemplate()
    {
        $this->validate([
            'template_subject' => 'required|string|max:255',
            'template_body' => 'required|string',
            'template_enabled' => 'boolean',
        ]);

        $template = EmailTemplate::findOrFail($this->editingTemplate);
        $template->update([
            'subject' => $this->template_subject,
            'body' => $this->template_body,
            'enabled' => $this->template_enabled,
        ]);

        session()->flash('template_success', 'Email template updated successfully!');

        $this->closeModal();
        $this->loadTemplates();
    }

    public function toggleEnabled($id)
    {
        $template = EmailTemplate::findOrFail($id);
        $template->update(['enabled' => !$template->enabled]);

        session()->flash('template_success', 'Template ' . ($template->enabled ? 'enabled' : 'disabled') . ' successfully!');

        $this->loadTemplates();
    }

    public function closeModal()
    {
        $this->showEditModal = false;
        $this->showPreviewModal = false;
        $this->reset(['editingTemplate', 'template_name', 'template_subject', 'template_body', 'template_enabled']);
    }

    public function getTemplateName($name)
    {
        return match ($name) {
            'welcome' => 'Welcome Email',
            'verify_email' => 'Email Verification',
            'reset_password' => 'Reset Password',
            'booking_success' => 'Booking Success',
            default => ucwords(str_replace('_', ' ', $name)),
        };
    }

    public function getPlaceholders($name)
    {
        $common = '{{site_name}}, {{site_url}}, {{logo_url}}, {{year}}, {{user_name}}';

        return match ($name) {
            'welcome' => $common,
            'verify_email' => $common . ', {{verification_url}}',
            'reset_password' => $common . ', {{reset_url}}',
            'booking_success' => $common . ', {{customer_name}}, {{invoice_number}}, {{listing_title}}, {{check_in}}, {{check_out}}, {{guests}}, {{total_price}}, {{host_name}}, {{host_email}}, {{host_phone}}, {{receipt_url}}',
            default => $common,
        };
    }

    public function savePaymentSettings()
    {
        $this->validate([
            'bayarcash_portal' => 'nullable|string|max:255',
            'bayarcash_pat' => 'nullable|string|max:1000',
            'bayarcash_secret' => 'nullable|string|max:255',
            'bayarcash_mode' => 'required|in:sandbox,production',
            'payment_gateway_fee_type' => 'required|in:fixed,percentage,both',
            'payment_gateway_fee_fixed' => 'required|numeric|min:0',
            'payment_gateway_fee_percentage' => 'required|numeric|min:0|max:100',
        ]);

        Setting::set('bayarcash_portal', $this->bayarcash_portal);
        Setting::set('bayarcash_pat', $this->bayarcash_pat);
        Setting::set('bayarcash_secret', $this->bayarcash_secret);
        Setting::set('bayarcash_mode', $this->bayarcash_mode);
        Setting::set('payment_gateway_fee_type', $this->payment_gateway_fee_type);
        Setting::set('payment_gateway_fee_fixed', $this->payment_gateway_fee_fixed);
        Setting::set('payment_gateway_fee_percentage', $this->payment_gateway_fee_percentage);

        session()->flash('payment_success', 'Payment gateway settings updated successfully!');
    }

    public function saveFeeSettings()
    {
        $this->validate([
            'platform_fee_type' => 'required|in:fixed,percentage,both',
            'platform_fee_fixed' => 'required|numeric|min:0',
            'platform_fee_percentage' => 'required|numeric|min:0|max:100',
            'platform_fee_charged_to' => 'required|in:guest,host',
            'featured_listing_fee' => 'required|numeric|min:0',
        ]);

        Setting::set('platform_fee_type', $this->platform_fee_type);
        Setting::set('platform_fee_fixed', $this->platform_fee_fixed);
        Setting::set('platform_fee_percentage', $this->platform_fee_percentage);
        Setting::set('platform_fee_charged_to', $this->platform_fee_charged_to);
        Setting::set('featured_listing_fee', $this->featured_listing_fee);

        session()->flash('fee_success', 'Platform fee settings updated successfully!');
    }

    public function saveWebhookSettings()
    {
        $this->validate([
            'webhook_url' => 'required_if:webhook_enabled,true|nullable|url|max:500',
            'webhook_secret_key' => 'nullable|string|max:255',
        ]);

        $webhookConfig = OutgoingWebhookConfig::getConfig();
        $webhookConfig->update([
            'enabled' => $this->webhook_enabled,
            'webhook_url' => $this->webhook_url,
            'payment_success' => $this->webhook_payment_success,
            'payment_failed' => $this->webhook_payment_failed,
            'booking_created' => $this->webhook_booking_created,
            'booking_cancelled' => $this->webhook_booking_cancelled,
            'secret_key' => $this->webhook_secret_key,
        ]);

        session()->flash('webhook_success', 'Webhook configuration updated successfully!');
    }

    public function saveEmailSettings()
    {
        $this->validate([
            'mail_mailer' => 'required|in:smtp,ses',
            'mail_host' => 'required_if:mail_mailer,smtp|nullable|string|max:255',
            'mail_port' => 'required_if:mail_mailer,smtp|nullable|integer|min:1|max:65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'nullable|in:tls,ssl',
            'mail_from_address' => 'required|email|max:255',
            'mail_from_name' => 'required|string|max:255',
            'aws_access_key_id' => 'required_if:mail_mailer,ses|nullable|string|max:255',
            'aws_secret_access_key' => 'required_if:mail_mailer,ses|nullable|string|max:255',
            'aws_default_region' => 'required_if:mail_mailer,ses|nullable|string|max:50',
        ]);

        Setting::set('mail_mailer', $this->mail_mailer);
        Setting::set('mail_host', $this->mail_host);
        Setting::set('mail_port', $this->mail_port);
        Setting::set('mail_username', $this->mail_username);
        Setting::set('mail_password', $this->mail_password);
        Setting::set('mail_encryption', $this->mail_encryption);
        Setting::set('mail_from_address', $this->mail_from_address);
        Setting::set('mail_from_name', $this->mail_from_name);
        Setting::set('aws_access_key_id', $this->aws_access_key_id);
        Setting::set('aws_secret_access_key', $this->aws_secret_access_key);
        Setting::set('aws_default_region', $this->aws_default_region);

        // Update mail configuration at runtime
        $this->updateMailConfig();

        session()->flash('email_success', 'Email settings updated successfully!');
    }

    protected function updateMailConfig()
    {
        if ($this->mail_mailer === 'ses') {
            config(['mail.default' => 'ses']);
            config(['services.ses.key' => $this->aws_access_key_id]);
            config(['services.ses.secret' => $this->aws_secret_access_key]);
            config(['services.ses.region' => $this->aws_default_region]);
        } else {
            config(['mail.default' => 'smtp']);
            config(['mail.mailers.smtp.host' => $this->mail_host]);
            config(['mail.mailers.smtp.port' => $this->mail_port]);
            config(['mail.mailers.smtp.username' => $this->mail_username]);
            config(['mail.mailers.smtp.password' => $this->mail_password]);
            config(['mail.mailers.smtp.encryption' => $this->mail_encryption]);
        }

        config(['mail.from.address' => $this->mail_from_address]);
        config(['mail.from.name' => $this->mail_from_name]);
    }

    public function saveReceiptSettings()
    {
        $this->validate([
            'receipt_template' => 'required|in:professional,compact',
            'receipt_company_name' => 'nullable|string|max:255',
            'receipt_company_address' => 'nullable|string|max:500',
            'receipt_company_phone' => 'nullable|string|max:50',
            'receipt_company_email' => 'nullable|email|max:255',
            'receipt_website' => 'nullable|url|max:255',
            'receipt_footer_text' => 'nullable|string|max:500',
            'receipt_show_logo' => 'boolean',
        ]);

        Setting::set('receipt_template', $this->receipt_template);
        Setting::set('receipt_company_name', $this->receipt_company_name);
        Setting::set('receipt_company_address', $this->receipt_company_address);
        Setting::set('receipt_company_phone', $this->receipt_company_phone);
        Setting::set('receipt_company_email', $this->receipt_company_email);
        Setting::set('receipt_website', $this->receipt_website);
        Setting::set('receipt_footer_text', $this->receipt_footer_text);
        Setting::set('receipt_show_logo', $this->receipt_show_logo);

        session()->flash('receipt_success', 'Receipt settings updated successfully!');
    }

    public function render()
    {
        return view('livewire.admin.settings')->layout('layouts.app');
    }
}
