<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\Setting;

class SeoSettings extends Component
{
    use WithFileUploads;

    // Basic SEO
    public $site_name;
    public $site_tagline;
    public $seo_title;
    public $seo_description;
    public $seo_keywords;

    // Images
    public $site_logo;
    public $site_favicon;
    public $og_image;
    public $new_logo;
    public $new_favicon;
    public $new_og_image;

    // Analytics & Verification
    public $google_analytics_id;
    public $google_site_verification;
    public $facebook_app_id;
    public $twitter_handle;

    // Theme & Display Settings
    public $logo_display_mode;
    public $theme_mode;
    public $custom_primary_color;

    // Footer Settings
    public $footer_copyright_text;
    public $footer_privacy_url;
    public $footer_terms_url;
    public $footer_sponsor_logos = [];
    public $new_sponsor_logo;
    public $sponsor_logo_to_delete;

    // URL Slugs
    public $privacy_policy_slug;
    public $terms_conditions_slug;
    public $advertise_slug;
    public $about_us_slug;

    // Privacy Policy & Terms Content
    public $privacy_policy_content;
    public $terms_conditions_content;

    // Advertise & About Us Content
    public $advertise_content;
    public $about_us_content;

    public function mount()
    {
        // Load existing settings
        $this->site_name = Setting::get('site_name', 'GengSewa');
        $this->site_tagline = Setting::get('site_tagline', 'Your Trusted Rental Marketplace');
        $this->seo_title = Setting::get('seo_title', 'GengSewa - Rent Cars, Boats, Houseboats & Guesthouses in Malaysia');
        $this->seo_description = Setting::get('seo_description', 'Discover and book the best rental properties in Malaysia.');
        $this->seo_keywords = Setting::get('seo_keywords', 'car rental malaysia, boat rental, houseboat rental');

        $this->site_logo = Setting::get('site_logo');
        $this->site_favicon = Setting::get('site_favicon');
        $this->og_image = Setting::get('og_image');

        $this->google_analytics_id = Setting::get('google_analytics_id');
        $this->google_site_verification = Setting::get('google_site_verification');
        $this->facebook_app_id = Setting::get('facebook_app_id');
        $this->twitter_handle = Setting::get('twitter_handle');

        $this->logo_display_mode = Setting::get('logo_display_mode', 'both');
        $this->theme_mode = Setting::get('theme_mode', 'system');
        $this->custom_primary_color = Setting::get('custom_primary_color', '#6366f1');

        $this->footer_copyright_text = Setting::get('footer_copyright_text', '© ' . date('Y') . ' GengSewa. All rights reserved.');
        $this->footer_privacy_url = Setting::get('footer_privacy_url', '/privacy-policy');
        $this->footer_terms_url = Setting::get('footer_terms_url', '/terms-and-conditions');

        $sponsorLogos = Setting::get('footer_sponsor_logos', '[]');
        $this->footer_sponsor_logos = is_string($sponsorLogos) ? json_decode($sponsorLogos, true) : $sponsorLogos;

        $this->privacy_policy_slug = Setting::get('privacy_policy_slug', '/privacy-policy');
        $this->terms_conditions_slug = Setting::get('terms_conditions_slug', '/terms-and-conditions');
        $this->advertise_slug = Setting::get('advertise_slug', '/advertise');
        $this->about_us_slug = Setting::get('about_us_slug', '/about-us');

        $this->privacy_policy_content = Setting::get('privacy_policy_content', '');
        $this->terms_conditions_content = Setting::get('terms_conditions_content', '');

        $this->advertise_content = Setting::get('advertise_content', '');
        $this->about_us_content = Setting::get('about_us_content', '');
    }

    public function save()
    {
        $this->validate([
            'site_name' => 'required|string|max:255',
            'site_tagline' => 'nullable|string|max:255',
            'seo_title' => 'required|string|max:255',
            'seo_description' => 'required|string|max:500',
            'seo_keywords' => 'nullable|string|max:500',
            'google_analytics_id' => 'nullable|string|max:50',
            'google_site_verification' => 'nullable|string|max:255',
            'facebook_app_id' => 'nullable|string|max:50',
            'twitter_handle' => 'nullable|string|max:50',
            'new_logo' => 'nullable|image|max:2048',
            'new_favicon' => 'nullable|image|mimes:ico,png|max:1024',
            'new_og_image' => 'nullable|image|max:2048',
            'logo_display_mode' => 'required|in:logo_only,name_only,both',
            'theme_mode' => 'required|in:system,dark,custom',
            'custom_primary_color' => 'required|regex:/^#[0-9A-Fa-f]{6}$/',
            'footer_copyright_text' => 'required|string',
            'footer_privacy_url' => 'required|string|max:255',
            'footer_terms_url' => 'required|string|max:255',
            'new_sponsor_logo' => 'nullable|image|max:2048',
            'privacy_policy_slug' => 'required|string|max:255',
            'terms_conditions_slug' => 'required|string|max:255',
            'advertise_slug' => 'required|string|max:255',
            'about_us_slug' => 'required|string|max:255',
            'privacy_policy_content' => 'required|string',
            'terms_conditions_content' => 'required|string',
            'advertise_content' => 'required|string',
            'about_us_content' => 'required|string',
        ]);

        // Handle logo upload
        if ($this->new_logo) {
            // Delete old logo if exists
            if ($this->site_logo && file_exists(public_path($this->site_logo))) {
                @unlink(public_path($this->site_logo));
            }

            // Create directory if it doesn't exist
            $logoDir = public_path('images' . DIRECTORY_SEPARATOR . 'logos');
            if (!is_dir($logoDir)) {
                mkdir($logoDir, 0777, true);
                chmod($logoDir, 0777);
            }

            $filename = time() . '_logo.' . $this->new_logo->getClientOriginalExtension();
            $tempPath = $this->new_logo->getRealPath();
            $destinationPath = $logoDir . DIRECTORY_SEPARATOR . $filename;

            // Use copy and unlink instead of move for better compatibility
            if (copy($tempPath, $destinationPath)) {
                @unlink($tempPath);
                chmod($destinationPath, 0644);
                $logoPath = 'images/logos/' . $filename;
                Setting::set('site_logo', $logoPath);
                $this->site_logo = $logoPath;
            }
        }

        // Handle favicon upload
        if ($this->new_favicon) {
            // Delete old favicon if exists
            if ($this->site_favicon && file_exists(public_path($this->site_favicon))) {
                @unlink(public_path($this->site_favicon));
            }

            // Create directory if it doesn't exist
            $faviconDir = public_path('images' . DIRECTORY_SEPARATOR . 'favicon');
            if (!is_dir($faviconDir)) {
                mkdir($faviconDir, 0777, true);
                chmod($faviconDir, 0777);
            }

            $filename = time() . '_favicon.' . $this->new_favicon->getClientOriginalExtension();
            $tempPath = $this->new_favicon->getRealPath();
            $destinationPath = $faviconDir . DIRECTORY_SEPARATOR . $filename;

            // Use copy and unlink instead of move for better compatibility
            if (copy($tempPath, $destinationPath)) {
                @unlink($tempPath);
                chmod($destinationPath, 0644);
                $faviconPath = 'images/favicon/' . $filename;
                Setting::set('site_favicon', $faviconPath);
                $this->site_favicon = $faviconPath;
            }
        }

        // Handle OG image upload
        if ($this->new_og_image) {
            // Delete old OG image if exists
            if ($this->og_image && file_exists(public_path($this->og_image))) {
                @unlink(public_path($this->og_image));
            }

            // Create directory if it doesn't exist
            $ogDir = public_path('images' . DIRECTORY_SEPARATOR . 'og-images');
            if (!is_dir($ogDir)) {
                mkdir($ogDir, 0777, true);
                chmod($ogDir, 0777);
            }

            $filename = time() . '_og.' . $this->new_og_image->getClientOriginalExtension();
            $tempPath = $this->new_og_image->getRealPath();
            $destinationPath = $ogDir . DIRECTORY_SEPARATOR . $filename;

            // Use copy and unlink instead of move for better compatibility
            if (copy($tempPath, $destinationPath)) {
                @unlink($tempPath);
                chmod($destinationPath, 0644);
                $ogImagePath = 'images/og-images/' . $filename;
                Setting::set('og_image', $ogImagePath);
                $this->og_image = $ogImagePath;
            }
        }

        // Save text settings
        Setting::set('site_name', $this->site_name);
        Setting::set('site_tagline', $this->site_tagline);
        Setting::set('seo_title', $this->seo_title);
        Setting::set('seo_description', $this->seo_description);
        Setting::set('seo_keywords', $this->seo_keywords);
        Setting::set('google_analytics_id', $this->google_analytics_id);
        Setting::set('google_site_verification', $this->google_site_verification);
        Setting::set('facebook_app_id', $this->facebook_app_id);
        Setting::set('twitter_handle', $this->twitter_handle);

        // Save theme and display settings
        Setting::set('logo_display_mode', $this->logo_display_mode);
        Setting::set('theme_mode', $this->theme_mode);
        Setting::set('custom_primary_color', $this->custom_primary_color);

        // Handle sponsor logo upload
        if ($this->new_sponsor_logo) {
            $logoDir = public_path('images' . DIRECTORY_SEPARATOR . 'sponsors');
            if (!is_dir($logoDir)) {
                mkdir($logoDir, 0777, true);
                chmod($logoDir, 0777);
            }

            $filename = time() . '_sponsor.' . $this->new_sponsor_logo->getClientOriginalExtension();
            $tempPath = $this->new_sponsor_logo->getRealPath();
            $destinationPath = $logoDir . DIRECTORY_SEPARATOR . $filename;

            if (copy($tempPath, $destinationPath)) {
                @unlink($tempPath);
                chmod($destinationPath, 0644);
                $logoPath = 'images/sponsors/' . $filename;
                $this->footer_sponsor_logos[] = $logoPath;
            }
        }

        // Save footer settings
        Setting::set('footer_copyright_text', $this->footer_copyright_text);
        Setting::set('footer_privacy_url', $this->footer_privacy_url);
        Setting::set('footer_terms_url', $this->footer_terms_url);
        Setting::set('footer_sponsor_logos', json_encode($this->footer_sponsor_logos));

        // Save URL slugs
        Setting::set('privacy_policy_slug', $this->privacy_policy_slug);
        Setting::set('terms_conditions_slug', $this->terms_conditions_slug);
        Setting::set('advertise_slug', $this->advertise_slug);
        Setting::set('about_us_slug', $this->about_us_slug);

        // Save privacy policy and terms content
        Setting::set('privacy_policy_content', $this->privacy_policy_content);
        Setting::set('terms_conditions_content', $this->terms_conditions_content);

        // Save advertise and about us content
        Setting::set('advertise_content', $this->advertise_content);
        Setting::set('about_us_content', $this->about_us_content);

        // Reset file inputs
        $this->new_logo = null;
        $this->new_favicon = null;
        $this->new_og_image = null;
        $this->new_sponsor_logo = null;

        session()->flash('message', 'SEO settings saved successfully!');
    }

    public function removeSponsorLogo($index)
    {
        if (isset($this->footer_sponsor_logos[$index])) {
            $logoPath = public_path($this->footer_sponsor_logos[$index]);
            if (file_exists($logoPath)) {
                @unlink($logoPath);
            }
            unset($this->footer_sponsor_logos[$index]);
            $this->footer_sponsor_logos = array_values($this->footer_sponsor_logos);
        }
    }

    public function render()
    {
        return view('livewire.admin.seo-settings')->layout('layouts.app');
    }
}
