<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\PickupLocation;

class PickupLocationsManager extends Component
{
    public $locations;
    public $showForm = false;
    public $editMode = false;
    public $locationId;

    public $name;
    public $address;
    public $type = 'general';
    public $latitude;
    public $longitude;
    public $is_active = true;
    public $typeFilter = '';

    public function mount()
    {
        $this->loadLocations();
    }

    public function loadLocations()
    {
        $query = PickupLocation::with('creator')->orderBy('created_at', 'desc');

        if ($this->typeFilter) {
            $query->where('type', $this->typeFilter);
        }

        $this->locations = $query->get();
    }

    public function updatedTypeFilter()
    {
        $this->loadLocations();
    }

    public function create()
    {
        $this->resetForm();
        $this->showForm = true;
        $this->editMode = false;
    }

    public function edit($id)
    {
        $location = PickupLocation::findOrFail($id);
        $this->locationId = $location->id;
        $this->name = $location->name;
        $this->address = $location->address;
        $this->type = $location->type;
        $this->latitude = $location->latitude;
        $this->longitude = $location->longitude;
        $this->is_active = $location->is_active;
        $this->showForm = true;
        $this->editMode = true;
    }

    public function save()
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string|max:500',
            'type' => 'required|in:airport,hotel,public_spot,office,general',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);

        $data = [
            'name' => $this->name,
            'address' => $this->address,
            'type' => $this->type,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'is_active' => $this->is_active,
        ];

        if ($this->editMode) {
            $location = PickupLocation::findOrFail($this->locationId);
            $location->update($data);
            session()->flash('message', 'Pickup location updated successfully!');
        } else {
            $data['created_by'] = auth()->id();
            PickupLocation::create($data);
            session()->flash('message', 'Pickup location created successfully!');
        }

        $this->loadLocations();
        $this->showForm = false;
        $this->resetForm();
    }

    public function toggleStatus($id)
    {
        $location = PickupLocation::findOrFail($id);
        $location->update(['is_active' => !$location->is_active]);
        $this->loadLocations();
        session()->flash('message', 'Location status updated!');
    }

    public function delete($id)
    {
        PickupLocation::findOrFail($id)->delete();
        $this->loadLocations();
        session()->flash('message', 'Pickup location deleted successfully!');
    }

    public function resetForm()
    {
        $this->name = '';
        $this->address = '';
        $this->type = 'general';
        $this->latitude = null;
        $this->longitude = null;
        $this->is_active = true;
    }

    public function render()
    {
        return view('livewire.admin.pickup-locations-manager')->layout('layouts.app');
    }
}
