<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Listing;
use App\Models\Setting;

class ListingsManager extends Component
{
    use WithPagination;

    public $perPage = 25;
    public $search = '';
    public $filterStatus = 'all'; // all, available, unavailable
    public $filterFeatured = 'all'; // all, featured, not_featured

    protected $queryString = ['search', 'filterStatus', 'filterFeatured'];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingFilterStatus()
    {
        $this->resetPage();
    }

    public function updatingFilterFeatured()
    {
        $this->resetPage();
    }

    public function toggleFeatured($listingId)
    {
        $listing = Listing::findOrFail($listingId);
        $listing->update(['is_featured' => !$listing->is_featured]);

        session()->flash('success', 'Listing featured status updated successfully!');
    }

    public function toggleAvailability($listingId)
    {
        $listing = Listing::findOrFail($listingId);
        $listing->update(['is_available' => !$listing->is_available]);

        session()->flash('success', 'Listing availability updated successfully!');
    }

    public function deleteListing($listingId)
    {
        $listing = Listing::findOrFail($listingId);
        $listing->delete();

        $this->resetPage(); // Reset pagination after delete
        session()->flash('success', 'Listing deleted successfully!');
    }

    public function render()
    {
        $query = Listing::with(['host', 'rentalType', 'location'])
            ->withCount(['bookings', 'reviews'])
            ->withAvg('reviews', 'rating');

        // Apply search filter
        if (!empty($this->search)) {
            $query->where(function ($q) {
                $q->where('title', 'like', '%' . $this->search . '%')
                    ->orWhere('description', 'like', '%' . $this->search . '%')
                    ->orWhereHas('host', function ($hostQuery) {
                        $hostQuery->where('name', 'like', '%' . $this->search . '%')
                            ->orWhere('email', 'like', '%' . $this->search . '%');
                    })
                    ->orWhereHas('location', function ($locQuery) {
                        $locQuery->where('name', 'like', '%' . $this->search . '%');
                    });
            });
        }

        // Apply status filter
        if ($this->filterStatus === 'available') {
            $query->where('is_available', true);
        } elseif ($this->filterStatus === 'unavailable') {
            $query->where('is_available', false);
        }

        // Apply featured filter
        if ($this->filterFeatured === 'featured') {
            $query->where('is_featured', true);
        } elseif ($this->filterFeatured === 'not_featured') {
            $query->where('is_featured', false);
        }

        $listings = $query->latest()->paginate($this->perPage);
        $featuredFee = Setting::get('featured_listing_fee', 0);

        return view('livewire.admin.listings-manager', [
            'listings' => $listings,
            'featuredFee' => $featuredFee,
        ])->layout('layouts.app');
    }
}
