<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\CustomPage;

class CustomPagesManager extends Component
{
    public $pages;
    public $editingPage = null;
    public $showModal = false;

    // Form fields
    public $title;
    public $slug;
    public $content;
    public $is_active = true;
    public $show_in_footer = true;
    public $order = 0;

    protected $rules = [
        'title' => 'required|string|max:255',
        'slug' => 'required|string|max:255|unique:custom_pages,slug',
        'content' => 'required|string',
        'is_active' => 'boolean',
        'show_in_footer' => 'boolean',
        'order' => 'integer|min:0',
    ];

    public function mount()
    {
        $this->loadPages();
    }

    public function loadPages()
    {
        $this->pages = CustomPage::orderBy('order')->get();
    }

    public function create()
    {
        $this->reset(['title', 'slug', 'content', 'is_active', 'show_in_footer', 'order', 'editingPage']);
        $this->is_active = true;
        $this->show_in_footer = true;
        $this->order = 0;
        $this->showModal = true;
    }

    public function edit($id)
    {
        $page = CustomPage::findOrFail($id);
        $this->editingPage = $page->id;
        $this->title = $page->title;
        $this->slug = $page->slug;
        $this->content = $page->content;
        $this->is_active = $page->is_active;
        $this->show_in_footer = $page->show_in_footer;
        $this->order = $page->order;
        $this->showModal = true;
    }

    public function save()
    {
        if ($this->editingPage) {
            $this->rules['slug'] = 'required|string|max:255|unique:custom_pages,slug,' . $this->editingPage;
        }

        $this->validate();

        if ($this->editingPage) {
            $page = CustomPage::findOrFail($this->editingPage);
            $page->update([
                'title' => $this->title,
                'slug' => $this->slug,
                'content' => $this->content,
                'is_active' => $this->is_active,
                'show_in_footer' => $this->show_in_footer,
                'order' => $this->order,
            ]);
            session()->flash('message', 'Page updated successfully!');
        } else {
            CustomPage::create([
                'title' => $this->title,
                'slug' => $this->slug,
                'content' => $this->content,
                'is_active' => $this->is_active,
                'show_in_footer' => $this->show_in_footer,
                'order' => $this->order,
            ]);
            session()->flash('message', 'Page created successfully!');
        }

        $this->showModal = false;
        $this->loadPages();
        $this->reset(['title', 'slug', 'content', 'is_active', 'show_in_footer', 'order', 'editingPage']);
    }

    public function delete($id)
    {
        CustomPage::findOrFail($id)->delete();
        session()->flash('message', 'Page deleted successfully!');
        $this->loadPages();
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->reset(['title', 'slug', 'content', 'is_active', 'show_in_footer', 'order', 'editingPage']);
        $this->resetValidation();
    }

    public function render()
    {
        return view('livewire.admin.custom-pages-manager')->layout('layouts.app');
    }
}
