<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Booking;
use Carbon\Carbon;

class BookingsManager extends Component
{
    use WithPagination;

    public $filter = 'all'; // all, pending, confirmed, completed, cancelled
    public $dateFilter = 'all'; // all, today, week, month, custom
    public $startDate = null;
    public $endDate = null;
    public $perPage = 10; // Pagination per page
    public $search = ''; // Search query

    // Analytics
    public $totalBookings = 0;
    public $confirmedBookings = 0;
    public $completedBookings = 0;
    public $totalRevenue = 0;

    public function mount()
    {
        $this->loadAnalytics();
    }

    public function getBookingsQuery()
    {
        $query = Booking::with(['listing.location', 'listing.rentalType', 'user', 'payment', 'addOns', 'pickupLocation', 'dropoffLocation'])
            ->orderBy('created_at', 'desc');

        // Apply search filter
        if (!empty($this->search)) {
            $query->where(function($q) {
                $q->where('id', 'like', '%' . $this->search . '%')
                  ->orWhere('invoice_number', 'like', '%' . $this->search . '%')
                  ->orWhereHas('user', function($q) {
                      $q->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('email', 'like', '%' . $this->search . '%');
                  })
                  ->orWhereHas('listing', function($q) {
                      $q->where('title', 'like', '%' . $this->search . '%');
                  });
            });
        }

        // Apply status filter
        if ($this->filter !== 'all') {
            $query->where('status', $this->filter);
        }

        // Apply date filter
        if ($this->dateFilter !== 'all') {
            switch ($this->dateFilter) {
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $query->whereBetween('created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()]);
                    break;
                case 'custom':
                    if ($this->startDate && $this->endDate) {
                        $query->whereBetween('created_at', [$this->startDate, $this->endDate]);
                    }
                    break;
            }
        }

        return $query;
    }

    public function loadAnalytics()
    {
        $query = Booking::query();

        // Apply date filter to analytics too
        if ($this->dateFilter !== 'all') {
            switch ($this->dateFilter) {
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $query->whereBetween('created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()]);
                    break;
                case 'custom':
                    if ($this->startDate && $this->endDate) {
                        $query->whereBetween('created_at', [$this->startDate, $this->endDate]);
                    }
                    break;
            }
        }

        $this->totalBookings = (clone $query)->count();
        $this->confirmedBookings = (clone $query)->where('status', 'confirmed')->count();
        $this->completedBookings = (clone $query)->where('status', 'completed')->count();
        $this->totalRevenue = (clone $query)->whereIn('status', ['confirmed', 'completed'])->sum('total_price');
    }

    public function updatedDateFilter()
    {
        $this->resetPage();
        $this->loadAnalytics();
    }

    public function updatedFilter()
    {
        $this->resetPage();
    }

    public function applyCustomDate()
    {
        if ($this->startDate && $this->endDate) {
            $this->dateFilter = 'custom';
            $this->resetPage();
            $this->loadAnalytics();
        }
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function approve($id)
    {
        $booking = Booking::findOrFail($id);
        $booking->update(['status' => 'confirmed']);
        $this->loadAnalytics();
        session()->flash('message', 'Booking approved successfully!');
    }

    public function complete($id)
    {
        $booking = Booking::findOrFail($id);
        $booking->update(['status' => 'completed']);
        $this->loadAnalytics();
        session()->flash('message', 'Booking marked as completed!');
    }

    public function cancel($id)
    {
        $booking = Booking::findOrFail($id);
        $booking->update(['status' => 'cancelled']);
        $this->loadAnalytics();
        session()->flash('message', 'Booking cancelled!');
    }

    public function delete($id)
    {
        $booking = Booking::findOrFail($id);
        $booking->delete();
        $this->loadAnalytics();
        session()->flash('message', 'Booking deleted successfully!');
    }

    public function exportCsv()
    {
        $bookings = $this->getBookingsQuery()
            ->with(['addOns', 'pickupLocation', 'dropoffLocation'])
            ->get();

        $filename = 'bookings_export_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($bookings) {
            $file = fopen('php://output', 'w');

            // Add CSV headers
            fputcsv($file, [
                'Invoice Number',
                'Guest Name',
                'Guest Email',
                'Listing Title',
                'Check In',
                'Check Out',
                'Guests',
                'Total Price (RM)',
                'Status',
                'Booking Date'
            ]);

            // Add data rows
            foreach ($bookings as $booking) {
                fputcsv($file, [
                    $booking->invoice_number,
                    $booking->user->name ?? 'N/A',
                    $booking->user->email ?? 'N/A',
                    $booking->listing->title ?? 'N/A',
                    $booking->check_in ? \Carbon\Carbon::parse($booking->check_in)->format('Y-m-d') : 'N/A',
                    $booking->check_out ? \Carbon\Carbon::parse($booking->check_out)->format('Y-m-d') : 'N/A',
                    $booking->guests,
                    number_format($booking->total_price, 2),
                    ucfirst($booking->status),
                    $booking->created_at->format('Y-m-d H:i:s')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function render()
    {
        $bookings = $this->getBookingsQuery()->paginate($this->perPage);

        return view('livewire.admin.bookings-manager', [
            'bookings' => $bookings
        ])->layout('layouts.app');
    }
}
