<?php

namespace App\Livewire\Admin;

use App\Models\BadgeConfig;
use Livewire\Component;
use Livewire\Attributes\On;

class BadgeConfigManager extends Component
{
    public $badges;
    public $editingId = null;
    public $badgeType = '';
    public $name = '';
    public $icon = '';
    public $color = '';
    public $minReviews = 0;
    public $minRating = 0;
    public $timePeriodMonths = 6;
    public $isActive = true;

    public function mount()
    {
        $this->loadBadges();
    }

    public function loadBadges()
    {
        $this->badges = BadgeConfig::orderBy('min_rating', 'desc')->get();
    }

    public function edit($id)
    {
        if ($id === 0) {
            // Creating a new badge
            $this->editingId = 0;
            $this->reset(['badgeType', 'name', 'icon', 'color', 'minReviews', 'minRating', 'timePeriodMonths']);
            $this->isActive = true;
            $this->color = 'bronze'; // Set default color
            $this->timePeriodMonths = 6; // Set default time period
        } else {
            // Editing existing badge
            $badge = BadgeConfig::find($id);
            if ($badge) {
                $this->editingId = $badge->id;
                $this->badgeType = $badge->badge_type;
                $this->name = $badge->name;
                $this->icon = $badge->icon;
                $this->color = $badge->color;
                $this->minReviews = $badge->min_reviews;
                $this->minRating = $badge->min_rating;
                $this->timePeriodMonths = $badge->time_period_months;
                $this->isActive = $badge->is_active;
            }
        }
    }

    public function save()
    {
        $this->validate([
            'badgeType' => 'required|string|max:255',
            'name' => 'required|string|max:255',
            'icon' => 'required|string|max:10',
            'color' => 'required|string|max:50',
            'minReviews' => 'required|integer|min:0',
            'minRating' => 'required|numeric|min:0|max:5',
            'timePeriodMonths' => 'required|integer|min:1',
        ]);

        if ($this->editingId === 0) {
            // Creating a new badge
            BadgeConfig::create([
                'badge_type' => $this->badgeType,
                'name' => $this->name,
                'icon' => $this->icon,
                'color' => $this->color,
                'min_reviews' => $this->minReviews,
                'min_rating' => $this->minRating,
                'time_period_months' => $this->timePeriodMonths,
                'is_active' => $this->isActive,
            ]);
        } else {
            // Updating existing badge
            BadgeConfig::updateOrCreate(
                ['id' => $this->editingId],
                [
                    'badge_type' => $this->badgeType,
                    'name' => $this->name,
                    'icon' => $this->icon,
                    'color' => $this->color,
                    'min_reviews' => $this->minReviews,
                    'min_rating' => $this->minRating,
                    'time_period_months' => $this->timePeriodMonths,
                    'is_active' => $this->isActive,
                ]
            );
        }

        $this->reset(['editingId', 'badgeType', 'name', 'icon', 'color', 'minReviews', 'minRating', 'timePeriodMonths', 'isActive']);
        $this->loadBadges();
        session()->flash('message', 'Badge configuration saved successfully.');
    }

    public function toggleActive($id)
    {
        $badge = BadgeConfig::find($id);
        if ($badge) {
            $badge->update(['is_active' => !$badge->is_active]);
            $this->loadBadges();
        }
    }

    public function delete($id)
    {
        BadgeConfig::find($id)?->delete();
        $this->loadBadges();
        session()->flash('message', 'Badge configuration deleted successfully.');
    }

    public function cancelEdit()
    {
        $this->reset(['editingId', 'badgeType', 'name', 'icon', 'color', 'minReviews', 'minRating', 'timePeriodMonths', 'isActive']);
    }

    public function render()
    {
        return view('livewire.admin.badge-config-manager')->layout('layouts.app');
    }
}
