<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\Amenity;
use App\Models\RentalType;

class AmenitiesManager extends Component
{
    public $amenities;
    public $showModal = false;
    public $editMode = false;

    // Form fields
    public $amenityId;
    public $name;
    public $rental_type_id;
    public $is_active = true;
    public $sort_order = 0;

    protected $rules = [
        'name' => 'required|string|max:255',
        'rental_type_id' => 'required|exists:rental_types,id',
        'is_active' => 'boolean',
        'sort_order' => 'integer|min:0',
    ];

    public function mount()
    {
        $this->loadAmenities();
    }

    public function loadAmenities()
    {
        $this->amenities = Amenity::with('rentalType')->orderBy('rental_type_id')->orderBy('sort_order')->orderBy('name')->get();
    }

    public function openModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->reset(['amenityId', 'name', 'rental_type_id', 'is_active', 'sort_order', 'editMode']);
        $this->is_active = true;
        $this->sort_order = 0;
        $this->resetValidation();
    }

    public function edit($id)
    {
        $amenity = Amenity::findOrFail($id);

        $this->amenityId = $amenity->id;
        $this->name = $amenity->name;
        $this->rental_type_id = $amenity->rental_type_id;
        $this->is_active = $amenity->is_active;
        $this->sort_order = $amenity->sort_order;
        $this->editMode = true;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        if ($this->editMode) {
            $amenity = Amenity::findOrFail($this->amenityId);
            $amenity->update([
                'name' => $this->name,
                'rental_type_id' => $this->rental_type_id,
                'is_active' => $this->is_active,
                'sort_order' => $this->sort_order,
            ]);
            session()->flash('message', 'Amenity updated successfully!');
        } else {
            Amenity::create([
                'name' => $this->name,
                'rental_type_id' => $this->rental_type_id,
                'is_active' => $this->is_active,
                'sort_order' => $this->sort_order,
            ]);
            session()->flash('message', 'Amenity created successfully!');
        }

        $this->closeModal();
        $this->loadAmenities();
    }

    public function toggleStatus($id)
    {
        $amenity = Amenity::findOrFail($id);
        $amenity->update(['is_active' => !$amenity->is_active]);
        $this->loadAmenities();
        session()->flash('message', 'Amenity status updated!');
    }

    public function delete($id)
    {
        $amenity = Amenity::findOrFail($id);
        $amenity->delete();
        $this->loadAmenities();
        session()->flash('message', 'Amenity deleted successfully!');
    }

    public function render()
    {
        return view('livewire.admin.amenities-manager', [
            'rentalTypes' => RentalType::all(),
        ])->layout('layouts.app');
    }
}
