<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;

class OptimizeProduction extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'production:optimize
                          {--clear : Clear all caches before optimizing}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize application for production (100-1000 daily requests)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Optimizing GengSewa for Production...');
        $this->newLine();

        // Clear caches if requested
        if ($this->option('clear')) {
            $this->warn('⚠️  Clearing all caches...');
            $this->clearAllCaches();
            $this->newLine();
        }

        // Step 1: Cache configuration
        $this->info('📝 Caching configuration...');
        Artisan::call('config:cache');
        $this->line('   ✓ Configuration cached');

        // Step 2: Cache routes
        $this->info('🛣️  Caching routes...');
        Artisan::call('route:cache');
        $this->line('   ✓ Routes cached');

        // Step 3: Cache views
        $this->info('👁️  Caching views...');
        Artisan::call('view:cache');
        $this->line('   ✓ Views cached');

        // Step 4: Cache events
        $this->info('📡 Caching events...');
        Artisan::call('event:cache');
        $this->line('   ✓ Events cached');

        // Step 5: Optimize autoloader
        $this->info('⚡ Optimizing Composer autoloader...');
        exec('composer dump-autoload -o 2>&1', $output, $returnCode);
        if ($returnCode === 0) {
            $this->line('   ✓ Autoloader optimized');
        } else {
            $this->warn('   ⚠️  Autoloader optimization skipped (composer not found or error)');
        }

        // Step 6: Build assets for production
        $this->info('🎨 Building production assets...');
        exec('npm run build 2>&1', $output, $returnCode);
        if ($returnCode === 0) {
            $this->line('   ✓ Assets built for production');
        } else {
            $this->warn('   ⚠️  Asset build skipped (npm not found or build error)');
        }

        $this->newLine();
        $this->info('✅ Production optimization complete!');
        $this->newLine();

        // Display optimization checklist
        $this->displayChecklist();
    }

    /**
     * Clear all application caches
     */
    protected function clearAllCaches()
    {
        Artisan::call('config:clear');
        $this->line('   ✓ Configuration cache cleared');

        Artisan::call('route:clear');
        $this->line('   ✓ Route cache cleared');

        Artisan::call('view:clear');
        $this->line('   ✓ View cache cleared');

        Artisan::call('event:clear');
        $this->line('   ✓ Event cache cleared');

        Artisan::call('cache:clear');
        $this->line('   ✓ Application cache cleared');
    }

    /**
     * Display post-optimization checklist
     */
    protected function displayChecklist()
    {
        $this->info('📋 Production Deployment Checklist:');
        $this->line('');
        $this->line('   [ ] Set APP_ENV=production in .env');
        $this->line('   [ ] Set APP_DEBUG=false in .env');
        $this->line('   [ ] Use Redis for cache, session, and queues');
        $this->line('   [ ] Configure queue worker: php artisan queue:work --tries=3');
        $this->line('   [ ] Set up cron job: * * * * * php artisan schedule:run');
        $this->line('   [ ] Enable OPcache in PHP');
        $this->line('   [ ] Configure database connection pooling');
        $this->line('   [ ] Set up HTTPS/SSL certificate');
        $this->line('   [ ] Configure firewall and security headers');
        $this->line('   [ ] Set up monitoring (logs, errors, performance)');
        $this->line('   [ ] Create database backups schedule');
        $this->line('');
        $this->info('💡 Tip: Run "php artisan production:optimize --clear" after code updates');
    }
}
