<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Cache;
use App\Models\Location;
use App\Models\RentalType;
use App\Models\Listing;
use App\Models\Setting;

class CacheWarmup extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cache:warmup';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Warm up application cache with frequently accessed data';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔥 Warming up application cache...');
        $this->newLine();

        // Cache locations with listing counts
        $this->info('📍 Caching locations...');
        Cache::remember('locations.all', 3600, function () {
            return Location::withCount('listings')->orderBy('name')->get();
        });
        $this->line('   ✓ Locations cached (1 hour TTL)');

        // Cache rental types
        $this->info('🏠 Caching rental types...');
        Cache::remember('rental_types.all', 3600, function () {
            return RentalType::all();
        });
        $this->line('   ✓ Rental types cached (1 hour TTL)');

        // Cache featured listings
        $this->info('⭐ Caching featured listings...');
        Cache::remember('listings.featured', 600, function () {
            return Listing::with(['location', 'rentalType'])
                ->withCount('reviews')
                ->withAvg('reviews', 'rating')
                ->featured()
                ->available()
                ->limit(6)
                ->get();
        });
        $this->line('   ✓ Featured listings cached (10 minutes TTL)');

        // Cache popular locations (top 10 by listing count)
        $this->info('🔝 Caching popular locations...');
        Cache::remember('locations.popular', 1800, function () {
            return Location::withCount('listings')
                ->having('listings_count', '>', 0)
                ->orderBy('listings_count', 'desc')
                ->limit(10)
                ->get();
        });
        $this->line('   ✓ Popular locations cached (30 minutes TTL)');

        // Cache application settings
        $this->info('⚙️  Caching application settings...');
        Cache::remember('settings.all', 3600, function () {
            return Setting::pluck('value', 'key')->toArray();
        });
        $this->line('   ✓ Settings cached (1 hour TTL)');

        // Cache available rental types with listing counts
        $this->info('📊 Caching rental type statistics...');
        Cache::remember('rental_types.stats', 1800, function () {
            return RentalType::withCount([
                'listings' => function ($query) {
                    $query->where('is_available', true);
                }
            ])->get();
        });
        $this->line('   ✓ Rental type stats cached (30 minutes TTL)');

        $this->newLine();
        $this->info('✅ Cache warmup complete!');
        $this->newLine();

        // Display cache status
        $this->displayCacheStatus();
    }

    /**
     * Display cache status information
     */
    protected function displayCacheStatus()
    {
        $this->info('📈 Cache Status:');
        $this->line('');
        $this->line('   Locations:        Cached (1 hour)');
        $this->line('   Rental Types:     Cached (1 hour)');
        $this->line('   Featured Listings: Cached (10 minutes)');
        $this->line('   Popular Locations: Cached (30 minutes)');
        $this->line('   Settings:         Cached (1 hour)');
        $this->line('   Type Statistics:  Cached (30 minutes)');
        $this->line('');
        $this->info('💡 Schedule this command to run every 30 minutes for optimal performance');
    }
}
