# GengSewa - Quick Production Setup Guide

## 🚀 Fast Track to Production (10 Minutes)

### Step 1: Environment Setup (2 minutes)
```bash
# Copy production config
cp .env.production.example .env

# Edit with your values (database, Redis, email, payment gateway)
nano .env

# Generate app key
php artisan key:generate
```

**Critical .env Settings:**
```env
APP_ENV=production
APP_DEBUG=false
DB_CONNECTION=mysql
CACHE_STORE=redis
QUEUE_CONNECTION=redis
SESSION_DRIVER=redis
```

### Step 2: Database Setup (3 minutes)
```bash
# Run all migrations (includes performance indexes)
php artisan migrate --force

# Verify indexes created
php artisan db:show
```

### Step 3: Production Optimization (2 minutes)
```bash
# Install dependencies
composer install --optimize-autoloader --no-dev
npm install && npm run build

# Optimize everything
php artisan production:optimize

# Warm up cache
php artisan cache:warmup
```

### Step 4: Queue Workers (2 minutes)
```bash
# Start queue worker
php artisan queue:work redis --tries=3 --timeout=60 &

# Or set up supervisor (recommended for production)
sudo nano /etc/supervisor/conf.d/gengsewa-worker.conf
```

**Supervisor Config:**
```ini
[program:gengsewa-worker]
command=php /var/www/gengsewa/artisan queue:work redis --tries=3
autostart=true
autorestart=true
user=www-data
numprocs=2
```

### Step 5: Schedule Tasks (1 minute)
```bash
# Add to crontab
crontab -e

# Add this line:
* * * * * cd /var/www/gengsewa && php artisan schedule:run >> /dev/null 2>&1
```

## ✅ Verification Checklist

After setup, verify everything works:

```bash
# Check database connection
php artisan db:show

# Check cache connection
php artisan tinker
>>> Cache::put('test', 'value', 60);
>>> Cache::get('test');

# Check queue connection
php artisan queue:work redis --once

# Check optimization status
php artisan optimize:clear  # Should clear all caches
php artisan route:cache     # Should cache routes
```

## 🔄 Daily Operations

### Deploy Updates
```bash
php artisan down
git pull
composer install --no-dev --optimize-autoloader
php artisan migrate --force
php artisan production:optimize --clear
php artisan cache:warmup
php artisan up
```

### Monitor Performance
```bash
# Check logs
tail -f storage/logs/laravel.log

# Check queue
php artisan queue:monitor redis

# Check failed jobs
php artisan queue:failed
```

### Clear Caches
```bash
# Clear specific caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Or clear all and re-optimize
php artisan production:optimize --clear
```

## 🐛 Common Issues & Fixes

### Cache not working
```bash
# Check Redis connection
redis-cli ping  # Should return PONG

# Restart Redis
sudo systemctl restart redis
```

### Queue jobs stuck
```bash
# Check worker status
sudo supervisorctl status gengsewa-worker:*

# Restart workers
sudo supervisorctl restart gengsewa-worker:*

# Process failed jobs
php artisan queue:retry all
```

### Slow performance
```bash
# Rebuild all caches
php artisan production:optimize --clear

# Warm up cache
php artisan cache:warmup

# Check database indexes
php artisan db:show
```

### Permission errors
```bash
# Fix storage permissions
sudo chown -R www-data:www-data storage bootstrap/cache
sudo chmod -R 775 storage bootstrap/cache
```

## 📊 Performance Expectations

With all optimizations applied:
- **Homepage**: < 200ms
- **Search**: < 300ms
- **Booking**: < 500ms
- **Database queries**: 5-15 per page (vs 50-100 unoptimized)
- **Cache hit ratio**: > 80%
- **Capacity**: 100-1000 daily bookings

## 📚 Full Documentation

For complete details, see:
- `PRODUCTION_DEPLOYMENT.md` - Full deployment guide
- `OPTIMIZATION_SUMMARY.md` - All optimizations applied
- `.env.production.example` - Environment template

## 🆘 Emergency Commands

### Site Down
```bash
php artisan down --retry=60  # Maintenance mode
php artisan up                # Bring back up
```

### Database Issues
```bash
php artisan db:show          # Show database info
php artisan migrate:status   # Check migrations
php artisan migrate:rollback # Rollback last migration
```

### Cache Issues
```bash
php artisan optimize:clear   # Nuclear option - clear everything
php artisan production:optimize  # Rebuild everything
```

## 💡 Pro Tips

1. **Always backup before updates**
   ```bash
   mysqldump -u user -p database > backup.sql
   ```

2. **Use maintenance mode for updates**
   ```bash
   php artisan down
   # ... do updates ...
   php artisan up
   ```

3. **Monitor queue workers**
   ```bash
   watch -n 1 'php artisan queue:monitor redis'
   ```

4. **Schedule cache warmup**
   ```bash
   # Add to app/Console/Kernel.php schedule():
   $schedule->command('cache:warmup')->everyThirtyMinutes();
   ```

5. **Keep logs clean**
   ```bash
   # Add to logrotate
   /var/www/gengsewa/storage/logs/*.log {
       daily
       rotate 14
       compress
   }
   ```

## ✨ That's It!

Your application is now production-ready and optimized for 100-1000 daily booking requests.

Need help? Check the full guides or Laravel documentation.
