# GengSewa - Complete Production Optimization Guide

This guide covers all optimizations for handling **3000+ bookings per month** with high performance.

## Table of Contents
1. [Database Optimization](#database-optimization)
2. [Laravel Application Optimization](#laravel-application-optimization)
3. [Caching Strategy](#caching-strategy)
4. [Asset Optimization](#asset-optimization)
5. [Server Configuration](#server-configuration)
6. [Monitoring & Maintenance](#monitoring--maintenance)

---

## 1. Database Optimization

### A. Apply Database Indexes
Database indexes are already configured in migration `2025_10_15_223948_add_performance_indexes_to_tables.php`

Run if not applied:
```bash
php artisan migrate
```

### B. Configure MySQL/MariaDB for Production

1. Copy the optimized configuration:
```bash
# Windows (Laragon)
cp docs/my.cnf.optimized C:/laragon/bin/mysql/mysql-X.X.X/my.ini

# Linux
cp docs/my.cnf.optimized /etc/mysql/my.cnf
```

2. Restart MySQL/MariaDB:
```bash
# Windows (Laragon) - Restart from Laragon GUI
# Linux
sudo systemctl restart mysql
```

3. Verify configuration:
```sql
SHOW VARIABLES LIKE 'innodb_buffer_pool_size';
SHOW VARIABLES LIKE 'max_connections';
```

### C. Database Maintenance Commands
```bash
# Optimize all tables monthly
php artisan tinker
DB::statement('OPTIMIZE TABLE bookings, listings, payments, users, reviews');

# Or via MySQL
mysql -u root -p gengsewa -e "OPTIMIZE TABLE bookings, listings, payments, users, reviews;"
```

### D. Query Optimization Checks
```bash
# Enable slow query log
# Check docs/my.cnf.optimized for slow query settings

# View slow queries
tail -f /var/log/mysql/slow-query.log
```

---

## 2. Laravel Application Optimization

### A. Essential Production Commands
Run these commands BEFORE deploying to production:

```bash
# 1. Install production dependencies only
composer install --optimize-autoloader --no-dev

# 2. Optimize autoloader
composer dump-autoload --optimize --classmap-authoritative

# 3. Cache configuration
php artisan config:cache

# 4. Cache routes
php artisan route:cache

# 5. Cache views
php artisan view:cache

# 6. Cache events
php artisan event:cache

# 7. Optimize (all-in-one)
php artisan optimize

# 8. Build frontend assets (production)
npm run build
```

### B. Environment Configuration (.env)
```bash
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yoursite.com

# Session optimization
SESSION_DRIVER=database
SESSION_LIFETIME=120

# Cache driver (Redis recommended for production)
CACHE_DRIVER=redis
QUEUE_CONNECTION=redis

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=gengsewa
DB_USERNAME=gengsewa_user
DB_PASSWORD=strong_password

# Redis (recommended for caching)
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
REDIS_CLIENT=phpredis

# Logging
LOG_CHANNEL=daily
LOG_LEVEL=warning
LOG_DEPRECATIONS_CHANNEL=null

# Mail (for production)
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=2525
```

### C. config/database.php Optimization
Add to your existing config:
```php
'mysql' => [
    // ... existing config ...
    'options' => extension_loaded('pdo_mysql') ? array_filter([
        PDO::MYSQL_ATTR_SSL_CA => env('MYSQL_ATTR_SSL_CA'),
        PDO::ATTR_PERSISTENT => true, // Connection pooling
        PDO::ATTR_EMULATE_PREPARES => false,
        PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => true,
    ]) : [],
    'dump' => [
        'dump_binary_path' => '/usr/bin', // Adjust for your system
    ],
    'sticky' => true, // Read-write separation if using replication
],
```

---

## 3. Caching Strategy

### A. Install Redis (Highly Recommended)
```bash
# Ubuntu/Debian
sudo apt-get install redis-server
sudo systemctl start redis
sudo systemctl enable redis

# Install PHP Redis extension
sudo apt-get install php-redis
sudo systemctl restart php8.2-fpm

# Windows (Laragon)
# Download Redis from https://github.com/microsoftarchive/redis/releases
# Or use Memcached which comes with Laragon
```

### B. Configure Redis in Laravel
Update `.env`:
```
CACHE_DRIVER=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis
```

### C. Implement Application Caching
The application already uses caching in:
- `app/Livewire/Home.php` - Caches locations, rental types, featured listings
- Other critical queries should follow the same pattern

Example pattern:
```php
use Illuminate\Support\Facades\Cache;

// Cache for 1 hour
$data = Cache::remember('cache_key', 3600, function () {
    return ExpensiveQuery::all();
});

// Cache with tags (Redis only)
$data = Cache::tags(['locations', 'active'])->remember('locations.all', 3600, function() {
    return Location::where('is_active', true)->get();
});
```

### D. Clear Cache Button
✅ Already implemented - Available in navigation header for admins

---

## 4. Asset Optimization

### A. CSS & JavaScript Optimization
```bash
# Install dependencies
npm install

# Build for production (minifies automatically)
npm run build

# Verify build output
ls -lh public/build/assets/
```

### B. Image Optimization

1. Install image optimization tools:
```bash
# Ubuntu/Debian
sudo apt-get install jpegoptim optipng pngquant gifsicle webp

# Windows - Install via Scoop or manually
```

2. Optimize existing images:
```bash
# JPEG
find public/images -name "*.jpg" -exec jpegoptim --max=85 {} \;

# PNG
find public/images -name "*.png" -exec optipng -o5 {} \;

# WebP conversion
find public/images -name "*.jpg" -exec cwebp -q 85 {} -o {}.webp \;
```

3. Set up automatic optimization (add to `.gitignore` if using CI/CD):
```bash
# Add to deployment script
jpegoptim --max=85 public/images/listings/*.jpg
optipng -o5 public/images/listings/*.png
```

### C. Enable Gzip/Brotli Compression
Add to `.htaccess` (Apache) or nginx config:

**Apache (.htaccess):**
```apache
# Enable compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript application/json
</IfModule>

# Enable browser caching
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/webp "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
</IfModule>
```

**Nginx:**
```nginx
# Gzip compression
gzip on;
gzip_vary on;
gzip_min_length 1024;
gzip_types text/plain text/css text/xml text/javascript application/javascript application/json application/xml+rss;

# Browser caching
location ~* \.(jpg|jpeg|png|gif|ico|css|js|webp)$ {
    expires 1y;
    add_header Cache-Control "public, immutable";
}
```

---

## 5. Server Configuration

### A. PHP-FPM Optimization
Edit `/etc/php/8.2/fpm/pool.d/www.conf`:

```ini
pm = dynamic
pm.max_children = 50
pm.start_servers = 10
pm.min_spare_servers = 5
pm.max_spare_servers = 20
pm.max_requests = 500

# Increase memory
php_admin_value[memory_limit] = 256M
```

Restart PHP-FPM:
```bash
sudo systemctl restart php8.2-fpm
```

### B. OPcache Configuration
Edit `/etc/php/8.2/fpm/conf.d/10-opcache.ini`:

```ini
opcache.enable=1
opcache.memory_consumption=256
opcache.interned_strings_buffer=16
opcache.max_accelerated_files=20000
opcache.revalidate_freq=60
opcache.fast_shutdown=1
opcache.enable_cli=0
```

### C. System Limits (Linux)
Edit `/etc/security/limits.conf`:
```
www-data soft nofile 65536
www-data hard nofile 65536
www-data soft nproc 32768
www-data hard nproc 32768
```

---

## 6. Monitoring & Maintenance

### A. Application Monitoring
```bash
# Check application logs
tail -f storage/logs/laravel.log

# Monitor queue workers
php artisan queue:work --tries=3 --timeout=60

# Check failed jobs
php artisan queue:failed
```

### B. Database Monitoring
```sql
-- Check table sizes
SELECT
    table_name,
    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS `Size (MB)`
FROM information_schema.TABLES
WHERE table_schema = 'gengsewa'
ORDER BY (data_length + index_length) DESC;

-- Check slow queries
SELECT * FROM mysql.slow_log ORDER BY query_time DESC LIMIT 10;

-- Monitor connections
SHOW PROCESSLIST;
SHOW STATUS LIKE '%connection%';
```

### C. Performance Testing
```bash
# Install Laravel Telescope (development only)
composer require laravel/telescope --dev
php artisan telescope:install
php artisan migrate

# Apache Bench testing
ab -n 1000 -c 10 https://yoursite.com/

# Laravel Debugbar (development only)
composer require barryvdh/laravel-debugbar --dev
```

### D. Regular Maintenance Schedule

**Daily:**
- Check error logs
- Monitor disk space
- Check database connections

**Weekly:**
- Review slow query log
- Clear old logs: `php artisan log:clear`
- Check queue status

**Monthly:**
- Optimize database tables
- Review and clean old data
- Update dependencies
- Security audit

---

## 7. Deployment Checklist

Use this checklist for every production deployment:

```bash
# 1. Pull latest code
git pull origin main

# 2. Install dependencies
composer install --optimize-autoloader --no-dev

# 3. Run migrations
php artisan migrate --force

# 4. Clear and rebuild caches
php artisan optimize:clear
php artisan optimize

# 5. Build assets
npm run build

# 6. Set permissions
chmod -R 755 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache

# 7. Restart services
sudo systemctl restart php8.2-fpm
sudo systemctl restart nginx

# 8. Clear OPcache (if needed)
php artisan config:clear && php artisan cache:clear

# 9. Test application
curl -I https://yoursite.com
```

---

## 8. Clean Up Unused Files

### A. Remove Unused Dependencies
```bash
# Remove development dependencies
composer install --no-dev

# Remove TinyMCE if not used (saves ~10MB)
# Only if you don't use the rich text editor
rm -rf public/vendor/tinymce

# Remove unused node modules after build
rm -rf node_modules
```

### B. Clean Laravel Files
```bash
# Clear old logs (older than 30 days)
find storage/logs -name "*.log" -type f -mtime +30 -delete

# Clear old cache files
php artisan cache:clear
php artisan view:clear

# Remove testing files in production
rm -rf tests/
rm phpunit.xml
```

---

## 9. Expected Performance Metrics

After implementing all optimizations:

- **Page Load Time:** < 500ms
- **Time to First Byte (TTFB):** < 200ms
- **Database Query Time:** < 50ms average
- **Concurrent Users:** 200-500
- **Requests per Second:** 100-200
- **Database Connections:** < 100 active
- **Memory Usage:** < 512MB per PHP process
- **CPU Usage:** < 50% under load

---

## 10. Quick Reference Commands

```bash
# Full optimization deployment
composer install --optimize-autoloader --no-dev
php artisan optimize
npm run build
php artisan migrate --force

# Clear all caches
php artisan optimize:clear

# Rebuild all caches
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache

# Check queue status
php artisan queue:work --tries=3

# Monitor logs
tail -f storage/logs/laravel.log

# Database optimization
php artisan db:optimize

# Clear cache via admin UI
# Log in as admin → Click cache icon in navigation header
```

---

## Support

For issues or questions:
- Check logs: `storage/logs/laravel.log`
- Database: Check slow query log
- Laravel Docs: https://laravel.com/docs
- Contact: support@gengsewa.com

---

**Last Updated:** 2025-10-17
**Version:** 1.0.0
